/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.ComponentModel;
using System.ComponentModel.Design;
using System.Drawing;
using System.Windows.Forms; // For BindingContext

using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Subscription;
using Borland.Eco.Services;
using Borland.Eco.Windows.Forms;
using Borland.Eco.Globalization;
using Borland.Eco.WinForm;

namespace Borland.Eco.Handles
{
	[ToolboxBitmap(typeof(CurrencyManagerHandle), "Borland.Eco.Handles.CurrencyManagerHandle.bmp")]
	[ToolboxItem(true)]
	[ToolboxItemFilter("System.Windows.Forms")]
	[Designer("Borland.Eco.Handles.Design.CurrencyManagerHandleDesigner, Borland.Eco.Handles.Design", typeof(IDesigner))]
	public sealed class CurrencyManagerHandle: AbstractCursorHandle
	{
		private Control bindingContext;
		/// <summary>
		/// Called if the BindingContext changes.
		/// </summary>
		public event EventHandler BindingContextChanged;
		private void OnBindingContextChanged()
		{
			if (BindingContextChanged != null)
				BindingContextChanged(this, EventArgs.Empty);
		}

		/// <summary>
		/// Set this property to the Control whose BindingContext shall be used.
		/// If the Default BindingContext is to be used, it can be set to the ContainerControl, which is usually the Form.
		/// </summary>
		[LocalizableCategory(typeof(FormsStringRes), "sCategoryConnections")]
		[LocalizableDescription(typeof(FormsStringRes), "sBindingContextControl")]
		[DefaultValue(null)]
		public Control BindingContext
		{
			get { return bindingContext; }
			set
			{
				bindingContext = value;
				OnBindingContextChanged();
				ElementChanged();
			}
		}

		/// <summary>
		/// <para>In a lot of cases users drop a CurrencyManagerHandle to get to the
		/// current element of a control.</para>
		///
		/// <para>This static method allows for retrieving that without having to
		/// instantiate a CurrencyManagerHandle.</para>
		/// It is also assumed that the control as a DataSource property pointing to an ElementHandle.
		/// CurrentElement(elementHandle, control) is then used to extract the actual object.
		/// </summary>
		public static IElement CurrentElement(Control control)
		{
			if (control == null) return null;
			ElementHandle elementHandle = EcoWinFormHelper.ConnectedHandle(control);
			return CurrentElement(elementHandle, control);
		}
		
		/// <summary>
		/// <para>In a lot of cases users drop a CurrencyManagerHandle to get to the
		/// current element of a control.</para>
		///
		/// <para>This static method allows for retrieving that without having to
		/// instantiate a CurrencyManagerHandle.</para>
		/// </summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="handle"/> is null.</exception>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		///<exception cref="ArgumentException">Thrown if no CurrencyManager can be found connecting the <paramref name="control"/> with the <paramref name="handle"/>.</exception>
		public static IElement CurrentElement(ElementHandle handle, Control control)
		{
			if (handle == null) throw new ArgumentNullException("handle"); // Do not localize
			if (control == null) throw new ArgumentNullException("control"); // Do not localize

			CurrencyManager cm = control.BindingContext[handle] as CurrencyManager;
			if (cm == null)
				throw new ArgumentException(FormsStringRes.sCannotFindBindingContext);
			IElement e = handle.Element;
			if (e is IElementCollection)
			{
				IElementCollection ec = handle.Element as IElementCollection;
				if ((cm.Position >= 0)  && (cm.Position < ec.Count))
					return ec[cm.Position];
				else
					return null;
			}
			else if (cm.Position == 0)
				return e;
			else
				return null;
		}

		private CurrencyManager m_currencyManager;

		private void FindAndListenToCurrencyManager()
		{
			CurrencyManager oldCurrencyManager = m_currencyManager;
			CurrencyManager newCurrencyManager = null;
			try
			{
				if (bindingContext == null)
					throw new InvalidOperationException(FormsStringRes.sCannotFindBindingContext);
				if ((RootHandle != null) && (bindingContext.BindingContext != null))
					newCurrencyManager = bindingContext.BindingContext[RootHandle] as CurrencyManager;
			}
			finally
			{
				if (oldCurrencyManager != newCurrencyManager)
				{
					EventHandler pce =  new EventHandler(PositionChanged);
					ItemChangedEventHandler ice = new ItemChangedEventHandler(ItemChanged);
					if (oldCurrencyManager != null)
					{
						oldCurrencyManager.PositionChanged -= pce;
						oldCurrencyManager.ItemChanged -= ice;
					}
					if (newCurrencyManager != null)
					{
						newCurrencyManager.PositionChanged += pce;
						newCurrencyManager.ItemChanged += ice;  // Note bug fix due to bug in listbox
					}
					m_currencyManager = newCurrencyManager;
				}
			}
		}

		/// <summary>
		/// <para>The currencyManager used by the Handle.</para>
		/// </summary>
		/// <remark>
		/// Note of bug in ListBox (or CurrencyManger). When the first element is created in a list,
		/// CurrencyManager will send a PositionChanged. The ListBox will raise an exception on this, since
		/// it has not yet updated its list to contain a element, and the expection will cut of notification to
		/// other controls subscribing to the same event.
		/// </remark>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public CurrencyManager CurrencyManager
		{
			get
			{
				if (m_currencyManager == null)
					FindAndListenToCurrencyManager();
				return m_currencyManager;
			}
		}

		private void PositionChanged(object sender, EventArgs e)
		{
			MarkSubscriptionOutOfDate();
		}

		private void ItemChanged(object sender, ItemChangedEventArgs e)
		{
			MarkSubscriptionOutOfDate();
		}

		public CurrencyManagerHandle(): base() {}

		protected override void Dispose(bool disposing)
		{
			try
			{
				if (m_currencyManager != null)
				{
					m_currencyManager.PositionChanged -= new EventHandler(PositionChanged);
					m_currencyManager = null;
				}
			}
			finally
			{
				base.Dispose(disposing);
			}
		}

		/// <summary>
		/// Setting the position will update the CurrencyManagers position, thus updating
		/// databound controls.
		/// </summary>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public override int Position
		{
			get
			{
				if (!DesignMode && (CurrencyManager != null)) //can't use CurrencyManager during design time!
					return CurrencyManager.Position;
				else
					return -1;
			}
			set
			{
				if (!DesignMode && (CurrencyManager != null)) //can't use CurrencyManager during design time!
					CurrencyManager.Position = value;;
			}
		}

		protected override void DeriveAndSubscribe(ISubscriber valueChangeSubscriber, ISubscriber resubscribeSubscriber)
		{
			if (Active)
			{
				IElement rootValue = EffectiveRootValue();
				IOclService oclService = (IOclService)GetEcoService(typeof(IOclService));
				if (oclService != null) 	// this is just to get the subscriptions
					oclService.EvaluateAndSubscribe(rootValue, EffectiveVariables(), "" /*Expression*/, valueChangeSubscriber, resubscribeSubscriber);

				if ((rootValue != null) &&
						(Position != -1))
				{
					IElementCollection ec = RootHandle.Element.GetAsCollection();
					if (ec.Count > CurrencyManager.Position)
						InternalElement = RootHandle.Element.GetAsCollection()[Position];
					else
						InternalElement = null;
				}
				else
					InternalElement = null;
			}
			else
			{
				InternalElement = null;
			}
		}
	}
}